// Makale Yazıcı - Ana JavaScript Dosyası
class ArticleGenerator {
    constructor() {
        this.quotes = [];
        this.apiKey = localStorage.getItem('openai_api_key') || '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';
        this.currentPreset = 'academic';
        this.temperature = 0.7; // Sabit değer
        this.maxTokens = parseInt(localStorage.getItem('maxTokens')) || 4000;
        this.customPrompt = localStorage.getItem('customPrompt') || `Aşağıda "xxx", "yyy" ve "zzz" olarak verilen cümleler, farklı akademik makalelerden veya tezlerden alınmış alıntı cümleleridir. Bu cümleleri doğrudan kopyalayıp kullanma. Bunun yerine, her birini dolaylı alıntı (parafraz) tekniğiyle, kendi cümlelerinmiş gibi yeniden yaz ve birbirleriyle uyumlu olacak şekilde tek bir bütünleşik paragraf haline getir. Paragraf, akıcı ve tutarlı bir akademik dilde yazılmalı.

⚠️ ÖNEMLİ KURALLAR:
1. TAM PARAFRAZ: Her alıntıyı tamamen farklı kelimelerle yeniden yaz. Hiçbir cümle orijinal metinle aynı olmasın.
2. KAYNAK BİLGİLERİNİ KORU: (Uzbay, 2015), (DiClemente, 2016) gibi kaynak referanslarını olduğu gibi koru. Sadece alıntı metinlerini parafraz et.
3. BAĞLAÇ KULLANIMI: Bağlaçlar kullanılabilir ama "sonra", "ama", "gibi", "ancak" gibi bağlaçlardan sonra virgül ekleme. Gereksiz geçiş ifadeleri kullanma.
4. DOĞAL AKIŞ: Cümleler doğal bir şekilde birbirine bağlansın, zorlamalı bağlaçlar kullanma.
5. AKADEMİK DİL: Resmî, nesnel, analitik bir üslup kullan ama sade ve anlaşılır olsun.
6. BÜTÜNLÜK: Alıntıları anlamlı şekilde birleştir, aralarına sadece gerekli açıklamalar ekle.
7. ÖZGÜNLÜK: Metin tamamen özgün olsun, orijinal metinlerle %0 benzerlik olsun.
8. AKIŞI BOZMA: Kaynak referanslarını koruyarak doğal akışı sağla.
9. ODAKLI YAKLAŞIM: Odak bir şekilde çıkar ve gerektiği kadar cümle kullan.
10. YAPAY ZEKA TESPİTİ: İnsan yazısı gibi doğal olsun, yapay zeka detektörüne yakalanmasın.

Beklenen çıktı:
- Odaklı ve tutarlı paragraf
- Tamamen özgün ifadeler
- Kaynak referansları korunur (Uzbay, 2015) gibi
- Bağlaçlar kullanılabilir
- Bağlaçlardan sonra virgül yok
- Doğal ve akıcı dil
- Akademik ama sade üslup
- Sadece alıntı metinleri parafraz edilir`;
        
        this.init();
    }

    init() {
        this.bindEvents();
        this.loadQuotesFromStorage();
        this.setupPresets();
        
        // API anahtarı varsayılan olarak ayarlanmış
        this.showMessage('MelTez hazır! Alıntıları ekleyip tez metni oluşturabil aşkım ❤️', 'success');
        
        // API key'i test et
        this.testApiKey();
    }

    bindEvents() {
        // Ana butonlar
        document.getElementById('addQuote').addEventListener('click', () => this.addQuote());
        document.getElementById('clearInput').addEventListener('click', () => this.clearInput());
        document.getElementById('generateArticle').addEventListener('click', () => this.generateArticle());
        
        // Makale işlemleri
        document.getElementById('copyArticle').addEventListener('click', () => this.copyArticle());
        document.getElementById('downloadArticle').addEventListener('click', () => this.downloadArticle());
        
        // Settings
        document.getElementById('settingsBtn').addEventListener('click', () => this.openSettings());
        document.getElementById('saveSettings').addEventListener('click', () => this.saveSettings());
        document.getElementById('resetSettings').addEventListener('click', () => this.resetSettings());
        
        
        // Modal kapatma
        document.querySelector('.close-btn').addEventListener('click', () => this.closeModal());
        
        // Enter tuşu ile alıntı ekleme
        document.getElementById('quoteInput').addEventListener('keydown', (e) => {
            if (e.ctrlKey && e.key === 'Enter') {
                this.addQuote();
            }
        });

        // Modal dışına tıklayınca kapatma
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('modal')) {
                this.closeModal();
            }
        });
    }

    setupPresets() {
        const defaultPrompt = `Aşağıda "xxx", "yyy" ve "zzz" olarak verilen cümleler, farklı akademik makalelerden veya tezlerden alınmış alıntı cümleleridir. Bu cümleleri doğrudan kopyalayıp kullanma. Bunun yerine, her birini dolaylı alıntı (parafraz) tekniğiyle, kendi cümlelerinmiş gibi yeniden yaz ve birbirleriyle uyumlu olacak şekilde tek bir bütünleşik paragraf haline getir. Paragraf, akıcı ve tutarlı bir akademik dilde yazılmalı.

⚠️ ÖNEMLİ KURALLAR:
1. TAM PARAFRAZ: Her alıntıyı tamamen farklı kelimelerle yeniden yaz. Hiçbir cümle orijinal metinle aynı olmasın.
2. KAYNAK BİLGİLERİNİ KORU: (Uzbay, 2015), (DiClemente, 2016) gibi kaynak referanslarını olduğu gibi koru. Sadece alıntı metinlerini parafraz et.
3. BAĞLAÇ KULLANIMI: Bağlaçlar kullanılabilir ama "sonra", "ama", "gibi", "ancak" gibi bağlaçlardan sonra virgül ekleme. Gereksiz geçiş ifadeleri kullanma.
4. DOĞAL AKIŞ: Cümleler doğal bir şekilde birbirine bağlansın, zorlamalı bağlaçlar kullanma.
5. AKADEMİK DİL: Resmî, nesnel, analitik bir üslup kullan ama sade ve anlaşılır olsun.
6. BÜTÜNLÜK: Alıntıları anlamlı şekilde birleştir, aralarına sadece gerekli açıklamalar ekle.
7. ÖZGÜNLÜK: Metin tamamen özgün olsun, orijinal metinlerle %0 benzerlik olsun.
8. AKIŞI BOZMA: Kaynak referanslarını koruyarak doğal akışı sağla.
9. ODAKLI YAKLAŞIM: Odak bir şekilde çıkar ve gerektiği kadar cümle kullan.
10. YAPAY ZEKA TESPİTİ: İnsan yazısı gibi doğal olsun, yapay zeka detektörüne yakalanmasın.

Beklenen çıktı:
- Odaklı ve tutarlı paragraf
- Tamamen özgün ifadeler
- Kaynak referansları korunur (Uzbay, 2015) gibi
- Bağlaçlar kullanılabilir
- Bağlaçlardan sonra virgül yok
- Doğal ve akıcı dil
- Akademik ama sade üslup
- Sadece alıntı metinleri parafraz edilir`;

        const presets = {
            academic: defaultPrompt
        };

        // Varsayılan prompt'u ayarla
        document.getElementById('customPrompt').value = defaultPrompt;
        
        // LocalStorage'a da kaydet
        localStorage.setItem('customPrompt', defaultPrompt);
    }

    addQuote() {
        const input = document.getElementById('quoteInput');
        const text = input.value.trim();
        
        if (!text) {
            this.showMessage('Lütfen bir alıntı girin!', 'error');
            return;
        }

        const quote = {
            id: Date.now(),
            text: text,
            order: this.quotes.length
        };

        this.quotes.push(quote);
        this.renderQuotes();
        this.saveQuotesToStorage();
        input.value = '';
        
        this.showMessage('Alıntı başarıyla eklendi!', 'success');
    }

    clearInput() {
        document.getElementById('quoteInput').value = '';
        document.getElementById('quoteInput').focus();
    }

    renderQuotes() {
        const container = document.getElementById('quotesList');
        
        if (this.quotes.length === 0) {
            container.innerHTML = `
                <div class="empty-state">
                    <i class="fas fa-quote-left"></i>
                    <p>Henüz alıntı eklenmedi</p>
                    <small>Sol panelden alıntı ekleyerek başla</small>
                </div>
            `;
            return;
        }

        container.innerHTML = this.quotes.map((quote, index) => `
            <div class="quote-item" data-id="${quote.id}">
                <div class="quote-text">"${quote.text}"</div>
                <div class="quote-actions">
                    <div class="quote-number">${index + 1}</div>
                    <div class="quote-controls">
                        <button class="control-btn move-up" onclick="articleGen.moveQuote(${quote.id}, 'up')" 
                                ${index === 0 ? 'disabled' : ''}>
                            <i class="fas fa-arrow-up"></i>
                        </button>
                        <button class="control-btn move-down" onclick="articleGen.moveQuote(${quote.id}, 'down')" 
                                ${index === this.quotes.length - 1 ? 'disabled' : ''}>
                            <i class="fas fa-arrow-down"></i>
                        </button>
                        <button class="control-btn delete-btn" onclick="articleGen.deleteQuote(${quote.id})">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </div>
            </div>
        `).join('');
    }

    moveQuote(id, direction) {
        const index = this.quotes.findIndex(q => q.id === id);
        if (index === -1) return;

        const newIndex = direction === 'up' ? index - 1 : index + 1;
        if (newIndex < 0 || newIndex >= this.quotes.length) return;

        // Swap quotes
        [this.quotes[index], this.quotes[newIndex]] = [this.quotes[newIndex], this.quotes[index]];
        
        this.renderQuotes();
        this.saveQuotesToStorage();
    }

    deleteQuote(id) {
        if (confirm('Bu alıntıyı silmek istediğinden emin misin?')) {
            this.quotes = this.quotes.filter(q => q.id !== id);
            this.renderQuotes();
            this.saveQuotesToStorage();
            this.showMessage('Alıntı silindi!', 'success');
        }
    }

    async generateArticle() {
        if (!this.apiKey) {
            this.showMessage('API anahtarı bulunamadı!', 'error');
            return;
        }

        if (this.quotes.length === 0) {
            this.showMessage('Lütfen önce alıntı ekle!', 'error');
            return;
        }

        this.showLoading(true);
        this.showMessage('Aşkım için tez paragrafı oluşturuluyor... ❤️', 'info');

        try {
            const customPrompt = document.getElementById('customPrompt').value.trim();
            const systemPrompt = customPrompt || this.getDefaultPrompt();
            
            // Dinamik alıntı formatı oluştur
            const quotePlaceholders = this.quotes.map((_, i) => `"${String.fromCharCode(120 + i)}${String.fromCharCode(120 + i)}${String.fromCharCode(120 + i)}"`).join(', ');
            const quotesText = this.quotes.map((q, i) => `"${String.fromCharCode(120 + i)}${String.fromCharCode(120 + i)}${String.fromCharCode(120 + i)}": "${q.text}"`).join('\n\n');
            
            const userPrompt = `
Aşağıda ${quotePlaceholders} olarak verilen cümleler, farklı akademik makalelerden veya tezlerden alınmış alıntı cümleleridir. Bu cümleleri doğrudan kopyalayıp kullanma. Bunun yerine, her birini dolaylı alıntı (parafraz) tekniğiyle, kendi cümlelerinmiş gibi yeniden yaz ve birbirleriyle uyumlu olacak şekilde tek bir bütünleşik paragraf haline getir. Paragraf, akıcı ve tutarlı bir akademik dilde yazılmalı.

Alıntılar:
${quotesText}

⚠️ ÖNEMLİ KURALLAR:
1. TAM PARAFRAZ: Her alıntıyı tamamen farklı kelimelerle yeniden yaz. Hiçbir cümle orijinal metinle aynı olmasın.
2. KAYNAK BİLGİLERİNİ KORU: (Uzbay, 2015), (DiClemente, 2016) gibi kaynak referanslarını olduğu gibi koru. Sadece alıntı metinlerini parafraz et.
3. BAĞLAÇ KULLANIMI: Bağlaçlar kullanılabilir ama "sonra", "ama", "gibi", "ancak" gibi bağlaçlardan sonra virgül ekleme. Gereksiz geçiş ifadeleri kullanma.
4. DOĞAL AKIŞ: Cümleler doğal bir şekilde birbirine bağlansın, zorlamalı bağlaçlar kullanma.
5. AKADEMİK DİL: Resmî, nesnel, analitik bir üslup kullan ama sade ve anlaşılır olsun.
6. BÜTÜNLÜK: Alıntıları anlamlı şekilde birleştir, aralarına sadece gerekli açıklamalar ekle.
7. ÖZGÜNLÜK: Metin tamamen özgün olsun, orijinal metinlerle %0 benzerlik olsun.
8. AKIŞI BOZMA: Kaynak referanslarını koruyarak doğal akışı sağla.
9. ODAKLI YAKLAŞIM: Odak bir şekilde çıkar ve gerektiği kadar cümle kullan.
10. YAPAY ZEKA TESPİTİ: İnsan yazısı gibi doğal olsun, yapay zeka detektörüne yakalanmasın.

Beklenen çıktı:
- Odaklı ve tutarlı paragraf
- Tamamen özgün ifadeler
- Kaynak referansları korunur (Uzbay, 2015) gibi
- Bağlaçlar kullanılabilir
- Bağlaçlardan sonra virgül yok
- Doğal ve akıcı dil
- Akademik ama sade üslup
- Sadece alıntı metinleri parafraz edilir
            `;

            const response = await this.callOpenAI(systemPrompt, userPrompt);
            this.displayArticle(response);
            
            this.showMessage('Tez metni başarıyla oluşturuldu!', 'success');
            
        } catch (error) {
            console.error('Tez metni oluşturma hatası:', error);
            this.showMessage('Tez metni oluşturulurken hata oluştu: ' + error.message, 'error');
        } finally {
            this.showLoading(false);
        }
    }

    async testApiKey() {
        try {
            const response = await fetch('https://api.openai.com/v1/models', {
                method: 'GET',
                headers: {
                    'Authorization': `Bearer ${this.apiKey}`
                }
            });
            
            if (response.ok) {
                console.log('✅ API Key çalışıyor!');
            } else {
                console.error('❌ API Key hatası:', response.status);
                this.showMessage('API anahtarı çalışmıyor! Lütfen ayarları kontrol et.', 'error');
            }
        } catch (error) {
            console.error('❌ API bağlantı hatası:', error);
            this.showMessage('API bağlantısı kurulamadı! İnternet bağlantını kontrol et.', 'error');
        }
    }

    async callOpenAI(systemPrompt, userPrompt) {
        const response = await fetch('https://api.openai.com/v1/chat/completions', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${this.apiKey}`
            },
            body: JSON.stringify({
                model: 'gpt-4o-2024-08-06',  // En güncel GPT-4 Omni modeli
                messages: [
                    {
                        role: 'system',
                        content: systemPrompt
                    },
                    {
                        role: 'user', 
                        content: userPrompt
                    }
                ],
                max_tokens: this.maxTokens,
                temperature: 0.3, // Daha düşük temperature = daha tutarlı
                top_p: 0.9,
                frequency_penalty: 0.2, // Tekrarları azalt
                presence_penalty: 0.1
            })
        });

        if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error?.message || 'API isteği başarısız oldu');
        }

        const data = await response.json();
        return data.choices[0].message.content;
    }

    getDefaultPrompt() {
        return `Sen deneyimli bir akademik editörsün. Verilen alıntıları parafraz tekniğiyle tamamen yeniden ifade ederek, akademik standartlarda bütünleşik bir metin oluştur. 
        Her alıntıyı kendi üslubunla yeniden yaz, anlamı koru ama ifadeyi değiştir. 
        Akademik dil kullan, bağlaçlar ve geçiş cümleleri ekle. Doğal insan yazısı gibi görünmesini sağla.`;
    }

    displayArticle(content) {
        const container = document.getElementById('generatedArticle');
        
        // Metni temizle ve düzenle
        const cleanedContent = this.cleanArticleContent(content);
        
        // Paragrafları ayır ve HTML formatına çevir
        const paragraphs = cleanedContent.split('\n\n').filter(p => p.trim());
        const htmlContent = paragraphs.map(p => `<p>${p.trim()}</p>`).join('');
        
        container.innerHTML = htmlContent;
        
        // Alt panele scroll yap
        container.scrollIntoView({ behavior: 'smooth' });
    }

    cleanArticleContent(content) {
        // Gereksiz ifadeleri temizle
        let cleaned = content
            .replace(/Bu durum,/g, '')
            .replace(/Bu nedenle,/g, '')
            .replace(/Bu bağlamda,/g, '')
            .replace(/Sonuç olarak,/g, '')
            .replace(/Bu genişletilmiş perspektif,/g, '')
            .replace(/Bu tanım,/g, '')
            .replace(/Bu süreç,/g, '')
            .replace(/Bu nedenle,/g, '')
            .replace(/\s+/g, ' ') // Çoklu boşlukları tek boşluğa çevir
            .trim();
        
        return cleaned;
    }

    copyArticle() {
        const content = document.getElementById('generatedArticle');
        const text = content.innerText;
        
        if (!text || text.includes('Tez metni burada görünecek')) {
            this.showMessage('Kopyalanacak tez metni bulunamadı!', 'error');
            return;
        }

        navigator.clipboard.writeText(text).then(() => {
            this.showMessage('Tez metni panoya kopyalandı!', 'success');
        }).catch(() => {
            this.showMessage('Kopyalama başarısız oldu!', 'error');
        });
    }

    downloadArticle() {
        const content = document.getElementById('generatedArticle');
        const text = content.innerText;
        
        if (!text || text.includes('Tez metni burada görünecek')) {
            this.showMessage('İndirilecek tez metni bulunamadı!', 'error');
            return;
        }

        const blob = new Blob([text], { type: 'text/plain;charset=utf-8' });
        const url = URL.createObjectURL(blob);
        
        const a = document.createElement('a');
        a.href = url;
        a.download = `tez_metni_${new Date().toISOString().split('T')[0]}.txt`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
        
        this.showMessage('Tez metni indirildi!', 'success');
    }

    openSettings() {
        document.getElementById('settingsModal').classList.add('active');
        document.getElementById('apiKey').value = this.apiKey;
        
        // Prompt'u her zaman tam olarak yükle
        const fullPrompt = `Aşağıda "xxx", "yyy" ve "zzz" olarak verilen cümleler, farklı akademik makalelerden veya tezlerden alınmış alıntı cümleleridir. Bu cümleleri doğrudan kopyalayıp kullanma. Bunun yerine, her birini dolaylı alıntı (parafraz) tekniğiyle, kendi cümlelerinmiş gibi yeniden yaz ve birbirleriyle uyumlu olacak şekilde tek bir bütünleşik paragraf haline getir. Paragraf, akıcı ve tutarlı bir akademik dilde yazılmalı.

⚠️ ÖNEMLİ KURALLAR:
1. TAM PARAFRAZ: Her alıntıyı tamamen farklı kelimelerle yeniden yaz. Hiçbir cümle orijinal metinle aynı olmasın.
2. KAYNAK BİLGİLERİNİ KORU: (Uzbay, 2015), (DiClemente, 2016) gibi kaynak referanslarını olduğu gibi koru. Sadece alıntı metinlerini parafraz et.
3. BAĞLAÇ KULLANIMI: Bağlaçlar kullanılabilir ama "sonra", "ama", "gibi", "ancak" gibi bağlaçlardan sonra virgül ekleme. Gereksiz geçiş ifadeleri kullanma.
4. DOĞAL AKIŞ: Cümleler doğal bir şekilde birbirine bağlansın, zorlamalı bağlaçlar kullanma.
5. AKADEMİK DİL: Resmî, nesnel, analitik bir üslup kullan ama sade ve anlaşılır olsun.
6. BÜTÜNLÜK: Alıntıları anlamlı şekilde birleştir, aralarına sadece gerekli açıklamalar ekle.
7. ÖZGÜNLÜK: Metin tamamen özgün olsun, orijinal metinlerle %0 benzerlik olsun.
8. AKIŞI BOZMA: Kaynak referanslarını koruyarak doğal akışı sağla.
9. ODAKLI YAKLAŞIM: Odak bir şekilde çıkar ve gerektiği kadar cümle kullan.
10. YAPAY ZEKA TESPİTİ: İnsan yazısı gibi doğal olsun, yapay zeka detektörüne yakalanmasın.

Beklenen çıktı:
- Odaklı ve tutarlı paragraf
- Tamamen özgün ifadeler
- Kaynak referansları korunur (Uzbay, 2015) gibi
- Bağlaçlar kullanılabilir
- Bağlaçlardan sonra virgül yok
- Doğal ve akıcı dil
- Akademik ama sade üslup
- Sadece alıntı metinleri parafraz edilir`;
        
        document.getElementById('customPrompt').value = fullPrompt;
        document.getElementById('maxTokens').value = this.maxTokens;
    }

    closeModal() {
        document.querySelectorAll('.modal').forEach(modal => {
            modal.classList.remove('active');
        });
    }

    saveSettings() {
        const apiKey = document.getElementById('apiKey').value.trim();
        
        if (!apiKey) {
            this.showMessage('Lütfen API anahtarını girin!', 'error');
            return;
        }

        if (!apiKey.startsWith('sk-')) {
            this.showMessage('Geçersiz API anahtarı formatı!', 'error');
            return;
        }

        // Ayarları kaydet
        this.apiKey = apiKey;
        this.customPrompt = document.getElementById('customPrompt').value.trim();
        this.maxTokens = parseInt(document.getElementById('maxTokens').value);

        // LocalStorage'a kaydet
        localStorage.setItem('openai_api_key', apiKey);
        localStorage.setItem('customPrompt', this.customPrompt);
        localStorage.setItem('maxTokens', this.maxTokens);

        this.closeModal();
        this.showMessage('Tüm ayarlar başarıyla kaydedildi!', 'success');
    }

    resetSettings() {
        if (confirm('Tüm ayarları sıfırlamak istediğinden emin misin?')) {
            // Varsayılan değerlere dön
            this.maxTokens = 4000;
            this.customPrompt = '';

            // LocalStorage'ı temizle (API key hariç)
            localStorage.removeItem('customPrompt');
            localStorage.removeItem('maxTokens');

            // Form'u güncelle
            document.getElementById('customPrompt').value = '';
            document.getElementById('maxTokens').value = 4000;

            this.showMessage('Ayarlar sıfırlandı!', 'success');
        }
    }

    showLoading(show) {
        const modal = document.getElementById('loadingModal');
        if (show) {
            modal.classList.add('active');
        } else {
            modal.classList.remove('active');
        }
    }

    showMessage(message, type = 'info') {
        // Eski mesajları temizle
        document.querySelectorAll('.message').forEach(msg => msg.remove());
        
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${type}`;
        messageDiv.textContent = message;
        
        // Header'dan sonra ekle
        const header = document.querySelector('.header');
        header.insertAdjacentElement('afterend', messageDiv);
        
        // 5 saniye sonra otomatik kaldır
        setTimeout(() => {
            messageDiv.remove();
        }, 5000);
    }

    saveQuotesToStorage() {
        localStorage.setItem('article_quotes', JSON.stringify(this.quotes));
    }

    loadQuotesFromStorage() {
        const saved = localStorage.getItem('article_quotes');
        if (saved) {
            this.quotes = JSON.parse(saved);
            this.renderQuotes();
        }
    }

    // Gelişmiş özellikler için hazırlık
    exportProject() {
        const projectData = {
            quotes: this.quotes,
            settings: {
                preset: this.currentPreset,
                customPrompt: document.getElementById('customPrompt').value
            },
            timestamp: new Date().toISOString()
        };
        
        const blob = new Blob([JSON.stringify(projectData, null, 2)], { 
            type: 'application/json' 
        });
        const url = URL.createObjectURL(blob);
        
        const a = document.createElement('a');
        a.href = url;
        a.download = `makale_projesi_${new Date().toISOString().split('T')[0]}.json`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }

    importProject(file) {
        const reader = new FileReader();
        reader.onload = (e) => {
            try {
                const projectData = JSON.parse(e.target.result);
                this.quotes = projectData.quotes || [];
                this.renderQuotes();
                this.saveQuotesToStorage();
                
                if (projectData.settings) {
                    this.currentPreset = projectData.settings.preset || 'creative';
                    document.getElementById('customPrompt').value = projectData.settings.customPrompt || '';
                    document.querySelector(`[data-preset="${this.currentPreset}"]`).click();
                }
                
                this.showMessage('Proje başarıyla yüklendi!', 'success');
            } catch (error) {
                this.showMessage('Proje dosyası okunamadı!', 'error');
            }
        };
        reader.readAsText(file);
    }
}

// Sayfa yüklendiğinde uygulamayı başlat
let articleGen;
document.addEventListener('DOMContentLoaded', () => {
    articleGen = new ArticleGenerator();
});

// Klavye kısayolları
document.addEventListener('keydown', (e) => {
    if (e.ctrlKey || e.metaKey) {
        switch(e.key) {
            case 'Enter':
                if (document.activeElement.id === 'quoteInput') {
                    e.preventDefault();
                    articleGen.addQuote();
                }
                break;
            case 's':
                e.preventDefault();
                articleGen.saveQuotesToStorage();
                articleGen.showMessage('Alıntılar kaydedildi!', 'success');
                break;
            case 'g':
                e.preventDefault();
                articleGen.generateArticle();
                break;
        }
    }
});

// Sayfa kapatılmadan önce uyarı
window.addEventListener('beforeunload', (e) => {
    if (articleGen && articleGen.quotes.length > 0) {
        e.preventDefault();
        e.returnValue = 'Kaydedilmemiş alıntılarınız var. Sayfayı kapatmak istediğinizden emin misiniz?';
    }
});
